/*
 * @(#)Tree.java
 */

package javax.ide.model.java.source.tree;

import java.util.List;
import java.util.ListIterator;
import javax.ide.model.java.source.SourcePosition;
import javax.ide.model.java.source.TreeVisitor;

/**
 * Base supertype of all AST nodes in the parse tree. <p/>
 *
 * @author Andy Yu
 * */
public interface Tree
{
  // ----------------------------------------------------------------------

  public static final Tree[] EMPTY_ARRAY = new Tree[ 0 ];


  // ----------------------------------------------------------------------

  /**
   * Identifies what kind of Tree this is. <p/>
   *
   * @return What kind of Tree this is.
   */
  public TreeKind getTreeKind();

  /**
   * Gets the position of this Tree object. If not synthetic, this
   * must always return non-null. If the position is unknown, zero may
   * be returned. <p/>
   *
   * The line and column must both be non-negative, but the URI may be
   * null. <p/>
   *
   * @return The position of this Tree object, null if this Tree
   * object is synthetic.
   */
  public SourcePosition getPosition();


  // ----------------------------------------------------------------------

  /**
   * True if this Tree element corresponds to an actual source
   * representation. <p/>
   * 
   * @return True if this Tree element corresponds to an actual source
   * representation. One example of a synthetic element is the default
   * constructor generated when a "class" or "enum" declaration does
   * not declare any constructors. Another example of a synthetic
   * element is the generated-on-the-fly InterfacesT element returned
   * by <code>ClassT.getInterfacesClause()</code> when there is no
   * interfaces declaration.
   */
  public boolean isSynthetic();


  // ----------------------------------------------------------------------

  /**
   * Gets the owning FileT.
   *
   * @return The owning FileT for this node.
   */
  public FileT getOwningFile();

  /**
   * Gets the parent Tree to this.
   *
   * @return The parent of this node.
   */
  public Tree getParent();

  /**
   * Gets the list of children Trees.
   *
   * @return The array of non-token children linked to this element. <p/>
   *
   * List of Trees.
   */
  public List getChildren();

  /**
   * Gets a ListIterator for this Tree's siblings.
   *
   * @return The same behavior as ListIterator. The "this" element
   * will be returned via an initial call to next(). If the initial
   * call is prev(), then the previous element will be returned. <p/>
   *
   * Note: Unlike ListIterator, next() and previous() will return null
   * instead of throwing NoSuchElementException's. That allows the client
   * to safely call getSiblings().previous() and getSiblings().next().
   */
  public ListIterator getSiblings();

  /**
   * Visits this element.
   */
  public void accept(TreeVisitor v);


  // ----------------------------------------------------------------------

  /**
   * Gets the sibling before this. <p/>
   *
   * This operation currently performs in time proportional to the
   * number of siblings this element has. That is, this is not a fast
   * operation.
   *
   * @return This element's sibling with an earlier start offset. Null
   * if none.
   */
  public Tree getSiblingBefore();

  /**
   * Gets the sibling after this. <p/>
   *
   * This operation currently performs in time proportional to the
   * number of siblings this element has. That is, this is not a fast
   * operation.
   *
   * @return This element's sibling with a later start offset. Null if
   * none.
   */
  public Tree getSiblingAfter();


  // ----------------------------------------------------------------------

  /**
   * Performs a deep-copy of this Tree but attached to the input
   * FileT. If this element has children, then the clone of this
   * element will have, as its children, clones of this element's
   * children. <p/>
   *
   * Properties are not copied. <p/>
   *
   * Cloning a FileT in this manner results in an
   * UnsupportedOperationException. To clone a FileT, please use
   * {@link javax.ide.model.java.source.TreeManager#cloneSourceFile(FileT,FileT)}. <p/>
   *
   * @return An unattached deep-copy of this Tree. "Deep copy" is
   *         explained above. "Unattached" means unparented.
   *
   * @throws UnsupportedOperationException if this is a FileT.
   */
  public Tree cloneSelf( FileT targetFile );

  /**
   * Performs an add (usually append) of this element to the parent.
   * Insertion point is left to the discretion of the implementation.
   *
   * @throws UnsupportedOperationException if this element may not
   * be added.
   * @throws IllegalStateException if this element already has a
   * parent.
   */
  public void addSelf( Tree parent );

  /**
   * Performs an add of this element to the parent of the input
   * sibling.
   *
   * @param before If true, this element will be added before the
   * sibling. If false, this element will be added after the sibling.
   *
   * @throws UnsupportedOperationException if addSelf does same.
   * @throws IllegalStateException if addSelf does same or if sibling
   * does not have a parent.
   */
  public void addSelf( Tree sibling, boolean before );

  /**
   */
  public void addSelfBefore( Tree sibling );

  /**
   */
  public void addSelfAfter( Tree sibling );

  /**
   * Performs a set of this element with the newElement.
   *
   * @throws UnsupportedOperationException if this element may not
   * be removed or the new element may not be added.
   * @throws IllegalStateException if this element does not have a
   * parent or if the new element already has a parent.
   */
  public void replaceSelf( Tree newElement );

  /**
   * Performs a remove of this element from its parent.
   *
   * @throws UnsupportedOperationException if this element may not
   * be removed.
   * @throws IllegalStateException if the element does not have a
   * parent.
   */
  public void removeSelf();


  // ----------------------------------------------------------------------

  /**
   * Gets the property saved with the given key, null if none. To prevent
   * collisions, clients should pick keys that start with the calling
   * class's {@link java.lang.Class#getName()} value. <p/>
   *
   * @return The property of the given type that is stored in this node,
   * null if none.
   */
  public Object getProperty(String key);

  /**
   * Stores the input property. Overwrites any property stored under the
   * same key.
   * 
   * @return The overwritten property, null if none.
   */
  public Object setProperty(String key, Object data);

  /**
   * Nulls out the property stored with the given key.
   *
   * @return The previously stored property, null if none.
   */
  public Object clearProperty(String key);


  // ----------------------------------------------------------------------
}
