/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;


/**
 * An element visitor that processes xml elements of type i18n_string. If
 * the element has an rskey attribute and an ancestor extension element defined
 * a bundle using the rsbundle attribute, this implementation will use
 * ResourceBundle.getBundle() to load a string resource from the bundle class.
 * If no rskey attribute is present, the text of the element will be used
 * instead. In either case, {@link #string( ElementContext, String )} is called
 * with the resulting string.<p>
 *
 * An error is logged if the rskey attribute is present on the element but no
 * rsbundle was defined in an ancestor extension element.
 *
 */
public abstract class I18NStringVisitor extends ElementVisitor
{
  protected static final String KEY_RSKEY = "rskey";

  public void start( ElementStartContext start )
  {
    String rskey = start.getAttributeValue( "rskey" );
    if ( rskey != null && ( rskey = rskey.trim()).length() == 0 )
    {
      // Must put null in the map so that we don't inherit rskey from parent
      // element.
      rskey = null;
    }
    
    // Check that we have an rsbundle. Otherwise, the use of rskey is flagged
    // as a warning.
    String rsBundleClass = (String) start.getScopeData().get( 
      ExtensionHook.KEY_RSBUNDLE_CLASS );
    if ( rskey != null && rsBundleClass == null )
    {
      log( start, Level.WARNING, 
        "'rskey' used without 'rsbundle' on extension element."
      );
      rskey = null;
    }
    start.getScopeData().put( KEY_RSKEY, rskey );
  }
  
  public void end( ElementEndContext end )
  {
    String rskey = (String) end.getScopeData().get( KEY_RSKEY );
    String value = null;
    if ( rskey != null )
    {
      try
      {
        value = lookupResource( end, rskey );
      }
      catch ( MissingResourceException mre )
      {      
        String bundleClass = (String) end.getScopeData().get( 
          ExtensionHook.KEY_RSBUNDLE_CLASS );
        log( end, Level.WARNING,
          "Resource key '"+rskey+"' not found in bundle '"+bundleClass+"'."
        );
        // Fall back to the text node, if any.
        value = null;
      }
    }
    
    if ( value == null )
    {
      value = end.getText();
      if ( value == null )
      {
        value = "";
      }
    }

    string( end, value.trim() );
  }
  
  private String lookupResource( ElementContext context, 
    String key )
  {
    final ResourceBundle bundle = ElementVisitor.getResourceBundle( context );
    if ( bundle == null )
    {
      return "!MB!" + key;
    }
    return bundle.getString( key );
  }
  
  /**
   * Called when the visitor has processed a translatable string.
   * 
   * @param context the context for parsing.
   * @param string the (possibly translated) string.
   */
  protected abstract void string( ElementContext context, String string );
}
