package javax.ide.editor;

/**
 * Base class for text based editors. Provides utilities for querying and 
 * changing the caret position and selection.
 */
public abstract class TextEditor extends Editor
{
  /**
   * Get the current caret position as a character offset from the start of the 
   * document.
   * 
   * @return the position of the caret as an offset from the start of the
   *  document.
   */
  public abstract int getCaretPosition();
  
  /**
   * Set the caret position. Any current selection will be cleared. If possible,
   * the editor is scrolled so that the caret is in view.
   * 
   * @param position the character offset from the start of the document at
   *    which to position the caret.
   * @throws java.lang.IndexOutOfBoundsException if the specified position
   *    is invalid.
   */
  public abstract void setCaretPosition( int position );
  
  /**
   * Set the caret position using a line and column number. Any current
   * selection will be cleared. If possible, the editor is scrolled so that
   * the caret is in view.
   * 
   * @param line the line number on which to position the caret. The first line
   *    in the document is 0.
   * @param column the column number at which to position the caret. The first
   *    column in the document is 0.
   * 
   * @throws java.lang.IndexOutOfBoundsException if the specified position is
   *    invalid.
   */
  public abstract void setCaretPosition( int line, int column );
  
  /**
   * Get the current column position of the caret.
   * 
   * @return the column position of the caret. If the caret is in the first
   *    column of the document, the return value will be <tt>0</tt>.
   */
  public abstract int getCaretColumn();
  
  /**
   * Get the current line position of the caret. 
   * 
   * @return the line position of the caret. This is zero-based; if the caret
   *    is on the first line in the document, the return value will be
   *    <tt>0</tt> not <tt>1</tt>.
   */
  public abstract int getCaretLine();
  
  /**
   * Get the character offset of the start of the current selection. If there
   * is no current selection, this returns the same value as
   * {@link #getCaretPosition()}.
   * 
   * @return the character offset of the start of the current selection.
   */
  public abstract int getSelectionStart();
  
  /**
   * Get the length of the current selection in characters. If there is no
   * current selection, this returns 0.
   * 
   * @return the number of characters currently selected.
   */
  public abstract int getSelectionLength();
  
  /**
   * Set the current selection. If possible, the selection is scrolled into
   * view.
   * 
   * @param start the start offset of the selection. This must be greater than
   *    0 and less than the length of the document-1.
   * @param length the length of the selection. This must not cause the 
   *    selection to extend past the end of the document.
   *    
   * @throws java.lang.IndexOutOfBoundsException if the specified selection
   *    position is invalid.
   */
  public abstract void setSelection( int start, int length );
}